odoo.define('account_invoice_extract.Box', function (require) {
"use strict";

var Widget = require('web.Widget');

/**
 * This widget represents a box that has been generated by the OCR. Such a box
 * is inserted on a box layer on a specific location. A box is related to
 * a field, and can be selected by the user and/or chosen by the OCR.
 */
var InvoiceExtractBox = Widget.extend({
    events: {
        'click': '_onClick'
    },
    template: 'account_invoice_extract.Box',
    /**
     * @override
     * @param {web.Class} parent class with EventDispatcherMixin
     * @param {Object} data
     * @param {float} data.box_angle angle in degrees for this box.
     * @param {float} data.box_height height of the box.
     * @param {float} data.box_midX x-coordinate of the middle point of this
     *   box.
     * @param {float} data.box_midY y-coordinate of the middle point of this
     *   box.
     * @param {float} data.box_width width of the box.
     * @param {string} data.feature name of the field that this box is related
     *   to.
     * @param {integer} data.id server-side ID of the box.
     * @param {integer} data.selected_status if not 0, this is chosen by the
     *   OCR.
     * @param {boolean} data.user_selected tell whether this box is selected by
     *   the user or not (server-side information).
     * @param {$.Element} data.$boxLayer jQuery element linked to the node of
     *   the box layer. This is used in order to compute the exact position of
     *   this box on the page.
     */
    init: function (parent, data) {
        this._super.apply(this, arguments);

        this._angle = data.box_angle;
        this._fieldName = data.feature;
        this._height = data.box_height;
        this._id = data.id;
        this._isOcrChosen = data.selected_status !== 0;
        this._isSelected = data.user_selected;
        this._midX = data.box_midX;
        this._midY = data.box_midY;
        this._text = data.text;
        this._width = data.box_width;
        this._$boxLayer = data.$boxLayer;
    },
    /**
     * Warn the invoice extract field that OCR chosen or selected boxes must be
     * tracked. This is useful in order to determine which box is selected.
     *
     * @override
     */
    start: function () {
        if (this._isSelected) {
            this.trigger_up('select_invoice_extract_box', {
                box: this,
            });
        }
        if (this._isOcrChosen) {
            this.trigger_up('choice_ocr_invoice_extract_box', {
                box: this,
            });
        }
        return this._super.apply(this, arguments);
    },
    /**
     * Warn the invoice extract field that this box should be untracked. This
     * is useful in order to keep using the same invoice extract field on
     * another invoice, which surely contains different boxes.
     *
     * @override
     */
    destroy: function () {
        this.trigger_up('destroy_invoice_extract_box', {
            box: this,
        });
        this._super.apply(this, arguments);
    },

    //--------------------------------------------------------------------------
    // Public
    //--------------------------------------------------------------------------

    /**
     * Get the field name of this box.
     *
     * @returns {string}
     */
    getFieldName: function () {
        return this._fieldName;
    },
    /**
     * Get the server-side ID of this box.
     *
     * @returns {integer}
     */
    getID: function () {
        return this._id;
    },
    /**
     * Get the style of this box, which is useful to represent the box at the
     * exact position and box size on the box layer.
     *
     * @returns {string}
     */
    getStyle: function () {
        return 'left:' + (this._midX * parseInt(this._$boxLayer[0].style.width)) + 'px;' +
                'top:' + (this._midY * parseInt(this._$boxLayer[0].style.height))  + 'px;' +
                'width:' + ((this._width) * parseInt(this._$boxLayer[0].style.width)) + 'px;' +
                'height:' + ((this._height) * parseInt(this._$boxLayer[0].style.height))  + 'px;' +
                'transform: translate(-50%, -50%) rotate(' + this._angle + 'deg);' +
                '-ms-transform: translate(-50%, -50%) rotate(' + this._angle + 'deg);' +
                '-webkit-transform: translate(-50%, -50%) rotate(' + this._angle + 'deg);';
    },
    /**
     * Tells whether the box has been chosen by the OCR or not.
     *
     * @returns {boolean}
     */
    isOcrChosen: function () {
        return this._isOcrChosen;
    },
    /**
     * @returns {boolean}
     */
    isSelected: function () {
        return this._isSelected;
    },
    /**
     * Set this box as 'selected'. This is called by the field that is tracking
     * boxes. Indeed, an OCR box without any user-selected box should be
     * selected.
     */
    setSelected: function () {
        if (!this._isSelected) {
            this._isSelected = true;
            this.$el.addClass('selected');
        }
    },
    /**
     * Unset this box as 'ocr chosen'. This case occurs when this box is
     * unselected when no other box is selected. This behaviour matches the
     * server-side behaviour.
     */
    unsetOcrChosen: function () {
        if (this._isOcrChosen) {
            this._isOcrChosen = false;
            this.$el.removeClass('ocr_chosen');
        }
    },
    /**
     * Unset this box as 'selected'. This is called by the field that is
     * tracking boxes. Indeed, an OCR box without any user-selected box should
     * be selected. When another box is selected by the user, the OCR chosen
     * box should no longer be selected.
     */
    unsetSelected: function () {
        if (this._isSelected) {
            this._isSelected = false;
            this.$el.removeClass('selected');
        }
    },

    //--------------------------------------------------------------------------
    // Handlers
    //--------------------------------------------------------------------------

    /**
     * Called when clicking on this box.
     *
     * @private
     * @param {MouseEvent} ev
     */
    _onClick: function (ev) {
        ev.stopPropagation();
        this.trigger_up('click_invoice_extract_box', {
            box: this,
        });
    },
});

return InvoiceExtractBox;
    
});

odoo.define('account_invoice_extract.BoxLayer', function (require) {
"use strict";

var InvoiceExtractBox = require('account_invoice_extract.Box');

var Widget = require('web.Widget');

/**
 * This widget is layer on top of a page in the attachment preview, in which
 * boxes are inserted. This widget handles boxes of a given page.
 */
var InvoiceExtractBoxLayer = Widget.extend({
    events: {
        'click': '_onClick',
    },
    className: 'boxLayer',
    /**
     * @override
     * @param {Class} parent a class with EventDispatcherMixin
     * @param {Object} data
     * @param {Object[]} data.boxesData list of all boxes data. Should filter
     *   on boxes that are linked to this box layer.
     * @param {string} data.mode either 'pdf' or 'img'
     * @param {integer} [data.pageNum=0]
     * @param {$.Element} [data.$buttons] mandatory in mode 'pdf': container of
     *   the field buttons, which is useful in adapt the height of box layer
     *   accordingly.
     * @param {$.Element} data.$page useful in order to auto resize box layer
     *   accordingly when attached to the DOM.
     * @param {$.Element} [data.$textLayer] mandatory in mode 'pdf': useful in
     *   order to size box layer similarly to text layer.
     */
    init: function (parent, data) {
        var self = this;
        this._super.apply(this, arguments);

        this._boxes = [];
        this._boxesData = data.boxesData;
        this._mode = data.mode;
        this._pageNum = data.pageNum || 0;

        this._$buttons = data.$buttons;
        this._$page = data.$page;
        this._$textLayer = data.$textLayer;

        // filter boxes data on current box layer
        this._boxesData = _.filter(this._boxesData, function (boxData) {
            return boxData.page === self._pageNum;
        });
    },
    /**
     * @override
     */
    start: function () {
        // adapt box layer size
        if (this._isOnPDF()) {
            this.el.style.width = this._$textLayer[0].style.width;
            this.el.style.height = this._$textLayer[0].style.height;
            this._$page[0].style.height = 'calc(100% - ' + this._$buttons.height() + 'px)';
        } else if (this._isOnImg()) {
            this.el.style.width = this._$page[0].clientWidth + 'px';
            this.el.style.height = this._$page[0].clientHeight + 'px';
            this.el.style.left = this._$page[0].offsetLeft + 'px';
            this.el.style.top = this._$page[0].offsetTop + 'px';
        }

        // make boxes (hidden by default)
        this._boxes = [];
        for (var index in this._boxesData) {
            var boxData = this._boxesData[index];
            var box = new InvoiceExtractBox(this, _.extend({}, boxData, {
                $boxLayer: this.$el,
            }));
            box.appendTo(this.$el);
            box.do_hide();
            this._boxes.push(box);
        }

        return this._super.apply(this, arguments);
    },

    //--------------------------------------------------------------------------
    // Public
    //--------------------------------------------------------------------------

    /**
     * Shows the boxes in this box layer, based on the selected field.
     *
     * @param {Object} params
     * @param {string} params.fieldName only show boxes of given field name
     */
    displayBoxes: function (params) {
        var selectedFieldName = params.fieldName;
        _.each(this._boxes, function (box) {
            if (box.getFieldName() === selectedFieldName) {
                box.do_show();
            } else {
                box.do_hide();
            }
        });
    },

    /**
     * Sets the textLayer for this box layer.
     *
     * @param {Object} params
     * @param {$.Element} [params.$textLayer] the new text layer
     */
    setTextLayer: function(params) {
        var $textLayer = params.$textLayer;
        this._$textLayer = $textLayer;
    },

    //--------------------------------------------------------------------------
    // Private
    //--------------------------------------------------------------------------

    /**
     * @private
     * @returns {boolean}
     */
    _isOnImg: function () {
        return this._mode === 'img';
    },
    /**
     * @private
     * @returns {boolean}
     */
    _isOnPDF: function () {
        return this._mode === 'pdf';
    },

    //--------------------------------------------------------------------------
    // Handlers
    //--------------------------------------------------------------------------

    /**
     * Called when clicking on the box layer.
     * This function should handle click away from a selected box.
     * This is ignored if the click on a box.
     *
     * @private
     * @param {MouseEvent} ev
     */
    _onClick: function (ev) {
        this.trigger_up('click_invoice_extract_box_layer');
    },
});

return InvoiceExtractBoxLayer;

});
    
odoo.define('account_invoice_extract.Field', function (require) {
"use strict";

var InvoiceExtractFieldButton = require('account_invoice_extract.FieldButton');

var Class = require('web.Class');
var field_utils = require('web.field_utils');
var Mixins = require('web.mixins');

/**
 * This class represents a field for the 'invoice extract' OCR feature. This is
 * useful in order to determine whether this feature is active or not, and also
 * to track some important boxes such as 'ocr chosen' and 'user selected' boxes.
 */
var InvoiceExtractField = Class.extend(Mixins.EventDispatcherMixin, {
    custom_events: {
        click_invoice_extract_field_button: '_onClickInvoiceExtractFieldButton'
    },
    /**
     * @override
     * @param {Class} parent a class with EventDispatcherMixin
     * @param {Object} params
     * @param {string} params.fieldName
     * @param {string} params.text
     */
    init: function (parent, params) {
        Mixins.EventDispatcherMixin.init.call(this, arguments);
        this.setParent(parent);

        this._button = undefined;
        this._isActive = false;
        this._name = params.fieldName;
        this._text = params.text;

        this._ocrChosenBox = undefined;
        this._selectedBox = undefined;
    },

    //--------------------------------------------------------------------------
    // Public
    //--------------------------------------------------------------------------

    /**
     * Get the selected box for this field
     *
     * @returns {account_invoice_extract.Box|undefined}
     */
    getSelectedBox: function () {
        return this._selectedBox;
    },
    /**
     * Get the field name
     *
     * @returns {string}
     */
    getName: function () {
        return this._name;
    },
    /**
     * Called to compute changes on a field, usually after selecting a new
     * box. The view should be updated with the new field info.
     *
     * @param {Object} params
     * @param {any} params.fieldChangedInfo
     * @param {Object} params.state state of the form renderer
     * @returns {Object}
     */
    handleFieldChanged: function (params) {
        var fieldChangedInfo = params.fieldChangedInfo;
        var state = params.state;
        var changes = {};
        switch (this._name) {
            case 'date':
                changes = { date_invoice: field_utils.parse.date(fieldChangedInfo.split(' ')[0]) };
                break;
            case 'supplier':
                if (_.isNumber(fieldChangedInfo)) {
                    changes = { partner_id: { id: fieldChangedInfo } };
                }
                break;
            case 'VAT_Number':
                changes = { partner_id: { id: fieldChangedInfo } };
                break;
            case 'due_date':
                changes = { date_due: field_utils.parse.date(fieldChangedInfo.split(' ')[0]) };
                break;
            case 'invoice_id':
                changes = { reference: fieldChangedInfo };
                break;
            case 'currency':
                changes = { currency_id: { id: fieldChangedInfo } };
                break;
        }
        return changes;
    },
    /**
     * Tell whether this field is active or not.
     *
     * @returns {boolean}
     */
    isActive: function () {
        return this._isActive;
    },
    /**
     * Render the button that is related to this field, so that the user can
     * select this field by clicking on the button.
     *
     * @param {Object} params
     * @param {$.Element} params.$container jQuery element which contains a
     *   single node in the DOM. This node is the container of the buttons.
     */
    renderButton: function (params) {
        this._button =  new InvoiceExtractFieldButton(this, {
            fieldName: this._name,
            isActive: this._isActive,
            text: this._text,
        });
        this._button.appendTo(params.$container);
    },
    /**
     * Set this field as active.
     */
    setActive: function () {
        if (!this._isActive) {
            this._isActive = true;
            if (this._button) {
                this._button.setActive();
            }
        }
    },
    /**
     * Set this field as inactive.
     */
    setInactive: function () {
        if (this._isActive) {
            this._isActive = false;
            if (this._button) {
                this._button.setInactive();
            }
        }
    },
    /**
     * Reset the boxes selected by user and ocr
     */
    resetSelection: function () {
        this._ocrChosenBox = undefined;
        this._selectedBox = undefined;
    },
    /**
     * Set the provided invoice extract 'box' as chosen by the OCR.
     *
     * @param {account_invoice_extract.Box} box
     */
    setOcrChosenBox: function (box) {
        this._ocrChosenBox = box;
        if (!this._selectedBox) {
            this._selectedBox = this._ocrChosenBox;
            this._ocrChosenBox.setSelected();
        }
    },
    /**
     * Set the provided invoice extract 'box' as selected.
     *
     * @param {account_invoice_extract.Box} box
     */
    setSelectedBox: function (box) {
        if (this._selectedBox) {
            this._selectedBox.unsetSelected();
        }
        this._selectedBox = box;
        if (this._ocrChosenBox && this._selectedBox !== this._ocrChosenBox) {
            this._ocrChosenBox.unsetSelected();
        }
        this._selectedBox.setSelected();
    },
    /**
     * Unselect the selected box. If the box to unselect is not the OCR chosen
     * box, make the ocr chosen box as selected. If the selected box is the
     * ocr chosen one, also remove the ocr chosen status of this box.
     */
    unselectBox: function () {
        if (!this._selectedBox) {
            return;
        }
        if (this._ocrChosenBox && this._ocrChosenBox.isSelected()) {
            this._selectedBox = undefined;
            this._ocrChosenBox.unsetSelected();
            this._ocrChosenBox.unsetOcrChosen();
            this._ocrChosenBox = undefined;
        } else if (this._ocrChosenBox) {
            this._selectedBox.unsetSelected();
            this._selectedBox = this._ocrChosenBox;
            this._ocrChosenBox.setSelected();
        } else {
            this._selectedBox.unsetSelected();
            this._selectedBox = undefined;
        }
    },
    /**
     * Remove tracking of this box. This method is called when the
     * corresponding box is destroyed.
     *
     * @param {account_invoice_extract.Box} box
     */
    unsetBox: function (box) {
        if (this._ocrChosenBox === this._selectedBox && this._ocrChosenBox === box) {
            this._ocrChosenBox = this._selectedBox = undefined;
        } else if (this._ocrChosenBox === box) {
            this._ocrChosenBox = undefined;
        } else {
            this._selectedBox = this._ocrSelectedBox;
        }
    },

    //--------------------------------------------------------------------------
    // Handlers
    //--------------------------------------------------------------------------

    /**
     * Called when the button corresponding to this field has been clicked.
     *
     * @private
     * @param {OdooEvent} ev
     */
    _onClickInvoiceExtractFieldButton: function (ev) {
        ev.stopPropagation();
        this.trigger_up('active_invoice_extract_field', {
            fieldName: this._name,
        });
    },
});

return InvoiceExtractField;

});
    
odoo.define('account_invoice_extract.FieldButton', function (require) {
"use strict";

var Widget = require('web.Widget');

/**
 * This widget represents a field button on top of the attachment preview,
 * which is used to filter boxes which the selected field.
 */
var InvoiceExtractFieldButton = Widget.extend({
    events: {
        'click': '_onClick',
    },
    template: 'account_invoice_extract.Button',
    /**
     * @override
     * @param {Class} parent a class with EventDispatcherMixin
     * @param {Object} params
     * @param {string} params.fieldName
     * @param {boolean} [params.isActive=false]
     * @param {string} params.text
     */
    init: function (parent, params) {
        this._super.apply(this, arguments);

        this._fieldName = params.fieldName;
        this._isActive = params.isActive || false;
        this._text = params.text;
    },

    //--------------------------------------------------------------------------
    // Public
    //--------------------------------------------------------------------------

    /**
     * Get the field name of the field button
     *
     * @returns {string}
     */
    getFieldName: function () {
        return this._fieldName;
    },
    /**
     * Get the text representation of the field button
     *
     * @returns {string}
     */
    getText: function () {
        return this._text;
    },
    /**
     * Tell whether this field button is active or not
     *
     * @returns {boolean}
     */
    isActive: function () {
        return this._isActive;
    },
    /**
     * Set this field button as active
     */
    setActive: function () {
        this._isActive = true;
        if (this.$el) {
            this.$el.addClass('active');
        }
    },
    /**
     * Set this field button as inactive
     */
    setInactive: function () {
        this._isActive = false;
        if (this.$el) {
            this.$el.removeClass('active');
        }
    },

    //--------------------------------------------------------------------------
    // Handlers
    //--------------------------------------------------------------------------

    /**
     * @private
     * @param {MouseEvent} ev
     */
    _onClick: function (ev) {
        ev.stopPropagation();
        this.trigger_up('click_invoice_extract_field_button');
    },
});

return InvoiceExtractFieldButton;

});
    
odoo.define('account_invoice_extract.Fields', function (require) {
"use strict";

var InvoiceExtractField = require('account_invoice_extract.Field');

var Class = require('web.Class');
var Mixins = require('web.mixins');
var ServicesMixin = require('web.ServicesMixin');
var session = require('web.session');

/**
 * This class groups the fields that are supported by the OCR. Also, it manages
 * the 'active' status of the fields, so that there is only one active field at
 * any given time.
 */
var InvoiceExtractFields = Class.extend(Mixins.EventDispatcherMixin, ServicesMixin, {
    custom_events: {
        active_invoice_extract_field: '_onActiveInvoiceExtractField',
    },
    /**
     * @override
     * @param {Class} parent a class with EventDispatcherMixin
     */
    init: function (parent) {
        var self = this;
        Mixins.EventDispatcherMixin.init.call(this, arguments);
        this.setParent(parent);

        this._fields = [
            new InvoiceExtractField(this, { text: 'VAT', fieldName: 'VAT_Number' }),
            new InvoiceExtractField(this, { text: 'Vendor', fieldName: 'supplier' }),
            new InvoiceExtractField(this, { text: 'Date', fieldName: 'date' }),
            new InvoiceExtractField(this, { text: 'Due Date', fieldName: 'due_date' }),
            new InvoiceExtractField(this, { text: 'Vendor Reference', fieldName: 'invoice_id' }),
        ];

        this._fields[0].setActive();
        session.user_has_group('base.group_multi_currency').then(function(has_multi_currency) {
            if (has_multi_currency) {
                self._fields.push(new InvoiceExtractField(self, { text: 'Currency', fieldName: 'currency' }));
            }
        });
    },

    //--------------------------------------------------------------------------
    // Public
    //--------------------------------------------------------------------------

    /**
     * Get the current active field.
     *
     * @returns {account_invoice_extract.Field|undefined} returns undefined if
     *   there is no active field.
     */
    getActiveField: function () {
        return _.find(this._fields, function (field) {
            return field.isActive();
        });
    },
    /**
     * Get the field with the given 'name'. If no field name is provided,
     * gets the active field.
     *
     * @param {Object} [params={}]
     * @param {string} [params.name] the field name
     * @returns {account_invoice_extract.Field|undefined} returns undefined if
     *   the provided field name does not exist.
     */
    getField: function (params) {
        params = params || {};
        if (!params.name) {
            return this.getActiveField();
        }
        return _.find(this._fields, function (field) {
            return field.getName() === params.name;
        });
    },
    /**
     * Render the buttons for each fields.
     *
     * @param {Object} params
     * @param {$.Element} params.$container jQuery element with a single node
     *   in the DOM, which is the container of the field buttons.
     */
    renderButtons: function (params) {
        _.each(this._fields, function (field) {
            field.renderButton(params);
        });
    },
    /**
     * Reset the active state of fields, so that the 1st field is active.
     */
    resetActive: function () {
        var oldActiveField = this.getActiveField();
        if (!oldActiveField) {
            return;
        }
        oldActiveField.setInactive();
        this._fields[0].setActive();
    },
    /**
     * Reset the active state of fields, so that the 1st field is active.
     */
    resetFieldsSelections: function () {
        _.each(this._fields, function (field) {
            field.resetSelection();
        });
    },

    //--------------------------------------------------------------------------
    // Handlers
    //--------------------------------------------------------------------------

    /**
     * Called when a field is selected (e.g. by clicking on the corresponding
     * button). This field becomes active, and other fields become inactive.
     *
     * @private
     * @param {OdooEvent} ev
     * @param {string} ev.data.fieldName
     */
    _onActiveInvoiceExtractField: function (ev) {
        var oldActiveField = this.getActiveField();
        if (!oldActiveField) {
            return;
        }
        oldActiveField.setInactive();
        var field = this.getField({ name: ev.data.fieldName });
        if (!field) {
            return;
        }
        field.setActive();
    },
});

return InvoiceExtractFields;

});
    
odoo.define('account_invoice_extract.FormRenderer', function (require) {
"use strict";

var InvoiceExtractBoxLayer = require('account_invoice_extract.BoxLayer');
var InvoiceExtractFields = require('account_invoice_extract.Fields');

var FormRenderer = require('web.FormRenderer');

/**
 * This is the renderer of the subview that adds OCR features on the attachment
 * preview. It displays boxes that have been generated by the OCR, and those
 * boxes are grouped by field name. The OCR automatically selects a box, but
 * the user can manually selects another box. Boxes are only visible in 'edit'
 * mode.
 */
var InvoiceExtractFormRenderer = FormRenderer.extend({
    custom_events: _.extend({}, FormRenderer.prototype.custom_events, {
        active_invoice_extract_field: '_onActiveInvoiceExtractField',
        preview_attachment_validation: '_onAttachmentPreviewValidation',
        choice_ocr_invoice_extract_box: '_onChoiceOcrInvoiceExtractBox',
        click_invoice_extract_box: '_onClickInvoiceExtractBox',
        click_invoice_extract_box_layer: '_onClickInvoiceExtractBoxLayer',
        destroy_invoice_extract_box: '_onDestroyInvoiceExtractBox',
        select_invoice_extract_box: '_onSelectInvoiceExtractBox',
    }),
    /**
     * @override
     */
    init: function () {
        this._super.apply(this, arguments);

        this._res_id = -1;
        this._invoiceExtractBoxData = [];
        this._invoiceExtractBoxLayers = [];
        this._invoiceExtractFields = new InvoiceExtractFields(this);
        this._$invoiceExtractButtons = [];
    },

    //--------------------------------------------------------------------------
    // Private
    //--------------------------------------------------------------------------

    /**
     * Remove any state related to the invoice extract, such as buttons or
     * boxes. This method is called when the form is not in 'edit' mode.
     *
     * @private
     */
    _destroyInvoiceExtract: function () {
        _.invoke(this._invoiceExtractBoxLayers, 'destroy');
        this._res_id = -1;
        this._invoiceExtractBoxData = [];
        this._invoiceExtractBoxLayers = [];
        this._invoiceExtractFields.resetActive();
        if (this._$invoiceExtractButtons.length) {
            this._$invoiceExtractButtons.remove();
        }
        this._$invoiceExtractButtons = [];
    },
    /**
     * Display the invoice extract boxes on the box layers for the active field.
     *
     * @private
     */
    _displayInvoiceExtractBoxes: function () {
        var field = this._invoiceExtractFields.getActiveField();
        if (!field) {
            return;
        }
        _.each(this._invoiceExtractBoxLayers, function (boxLayer) {
            boxLayer.displayBoxes({ fieldName: field.getName() });
        });
    },
    /**
     * Called when a box or box layer interaction has been performed by the
     * user (such as selecting a new box). Usually, a field in the form view
     * should be updated with the box content after clicking on the box.
     *
     * @private
     * @param {Object} params
     * @param {any} params.fieldChangedInfo new info on the field, based on
     *   box or box layer interaction
     */
    _handleInvoiceExtractFieldChanged: function (params) {
        var fieldChangedInfo = params.fieldChangedInfo;
        var field = this._invoiceExtractFields.getActiveField();
        if (!field) {
            return;
        }
        var changes = field.handleFieldChanged({
            fieldChangedInfo: fieldChangedInfo,
            state: this.state,
        });
        if (!_.isEmpty(changes)) {
            this.trigger_up('field_changed', {
                dataPointID: this.state.id,
                changes: changes,
            });
        }
    },
    /**
     * Setup data for the invoice extract stuffs. In particular, get box data
     * and render them. Boxes are displayed on the provided page.
     *
     * @private
     * @param {$.Element} $page jQuery element corresponding to the attachment
     *   page. This is useful for the rendering of the box layer, which is
     *   handled differently whether the attachment is an image or a pdf.
     */
    _initInvoiceExtract: function ($page) {
        var self = this;
        this._res_id = this.state.res_id;
        this._rpc({
            model: 'account.invoice',
            method: 'get_boxes',
            args: [this.state.res_id],
        }).then(function (boxesData) {
            self._invoiceExtractBoxData = boxesData;
            self._renderInvoiceExtractBoxLayers({
                $page: $page,
            });
            self._displayInvoiceExtractBoxes();
        });
    },
    /**
     * Render the field buttons on top of the attachment page preview, so that
     * the user can select the boxes to show for the given field.
     *
     * @private
     */
    _renderInvoiceExtractButtons: function () {
        this._invoiceExtractFields.renderButtons({
            $container: this._$invoiceExtractButtons,
        });
    },
    /**
     * Render the box layers using fetched box data and the attachment page
     * preview. Boxes data are used in order to make and render boxes, whereas
     * the page is used in order to correctly size the box layers. Also, the
     * setup of the box layers slightly differs with images and pdf, because
     * the latter uses an iframe.
     *
     * Because the iframe is created with the pdf viewer, and there is no odoo
     * bundle related to it, we cannot simply extend the bundle with SCSS
     * styles. As a work-around, the style of the boxes and box layers must be
     * defined in CSS, so that we can dynamically add this file in the header
     * of the iframe (for the case of PDF viewer).
     *
     * @private
     * @param {Object} params
     * @param {$.Element} params.$page the attachment page preview container
     */
    _renderInvoiceExtractBoxLayers: function (params) {
        var $page = params.$page;
        var boxLayer;
        //in case of img
        if ($page.hasClass('img-fluid')) {
            // dynamically add css on the image (not in asset bundle)
            $('head').append('<link rel="stylesheet" type="text/css" href="/account_invoice_extract/static/src/css/account_invoice_extract.css"></link>');
            boxLayer = new InvoiceExtractBoxLayer(this, {
                boxesData: this._invoiceExtractBoxData,
                mode: 'img',
                $page: $page,
            });
            boxLayer.insertAfter($page);
            this._invoiceExtractBoxLayers = [boxLayer];
        }
        //in case of pdf
        if ($page.is('iframe')) {
            var $document = $page.contents();
            // dynamically add css on the pdf viewer
            $document.find('head').append('<link rel="stylesheet" type="text/css" href="/account_invoice_extract/static/src/css/account_invoice_extract.css"></link>');
            var $textLayers = $document.find('.textLayer');
            for (var index = 0; index < $textLayers.length; index++) {
                var $textLayer = $textLayers.eq(index);
                var pageNum = $textLayer[0].parentElement.dataset['pageNumber'] - 1;
                var existingBoxLayer = this._invoiceExtractBoxLayers.find(function(bl) {
                    return bl._pageNum == pageNum;
                });
                if (existingBoxLayer) {
                    existingBoxLayer.setTextLayer({
                        $textLayer: $textLayer,
                    });
                    existingBoxLayer.insertAfter($textLayer);
                }
                else {
                    boxLayer = new InvoiceExtractBoxLayer(this, {
                        boxesData: this._invoiceExtractBoxData,
                        mode: 'pdf',
                        pageNum: pageNum,
                        $buttons: this._$invoiceExtractButtons,
                        $page: $page,
                        $textLayer: $textLayer,
                    });
                    boxLayer.insertAfter($textLayer);
                    this._invoiceExtractBoxLayers.push(boxLayer);
                }
            }
        }
    },
    /**
     * @private
     * @param {$.Element} $attachment
     */
    _startInvoiceExtract: function ($attachment) {
        var $ocrSuccess = this.$('.o_success_ocr');
        if (
            $attachment.length !== 0 &&
            $ocrSuccess.length > 0 &&
            !$ocrSuccess.hasClass('o_invisible_modifier') &&
            this.mode === 'edit'
        ) {
            // fetch boxes data for this record
            if (this._res_id != this.state.res_id) {
                this._destroyInvoiceExtract();
                this._initInvoiceExtract($attachment);
            }
            // render boxes if their data has already been fetched
            else if (this._invoiceExtractBoxData.length > 0) {
                this._renderInvoiceExtractBoxLayers({
                    $page: $attachment,
                })
                this._displayInvoiceExtractBoxes();
            }
            // render buttons if they're not already rendered
            if (this._$invoiceExtractButtons.length == 0) {
                this._$invoiceExtractButtons = $('<div class="o_invoice_extract_buttons"/>');
                $attachment.before(this._$invoiceExtractButtons);
                this._renderInvoiceExtractButtons();
            }
        } else {
            this._destroyInvoiceExtract();
        }
    },

    //--------------------------------------------------------------------------
    // Handlers
    //--------------------------------------------------------------------------

    /**
     * Called when there is a change on active field. It should display boxes
     * related to the newly active field.
     *
     * @private
     * @param {OdooEvent} ev
     */
    _onActiveInvoiceExtractField: function (ev) {
        ev.stopPropagation();
        this._displayInvoiceExtractBoxes();
    },
    /**
     * Called when the chatter has rendered the attachment preview. In 'edit'
     * mode, it should render the invoice extract boxes and buttons.
     *
     * @private
     * @param {OdooEvent} ev
     */
    _onAttachmentPreviewValidation: function (ev) {
        ev.stopPropagation();
        var self = this;
        if (this.$attachmentPreview === undefined) {
            return;
        }
        // case pdf (iframe)
        this.$attachmentPreview.find('iframe').load(function () { // wait for iframe to load
            var $iframe = self.$attachmentPreview.find('iframe');
            var $iframeDoc = self.$attachmentPreview.find('iframe').contents();
            $iframeDoc[0].addEventListener('pagerendered', function () {
                self._startInvoiceExtract($iframe);
            });
        });
        // case img
        var $documentImg = self.$attachmentPreview.find('.img-fluid');
        var $attachment = $('#attachment_img');
        if ($attachment.length && $attachment[0].complete) {
            this._startInvoiceExtract($documentImg);
        } else {
            this.$attachmentPreview.find('.img-fluid').load(function () {
                self._startInvoiceExtract($documentImg);
            });
        }
    },
    /**
     * Called when clicking on a box. In this case, the box should be selected,
     * and the field in the form should be updated accordingly.
     *
     * @private
     * @param {OdooEvent} ev
     * @param {account_invoice_extract.Box} ev.data.box
     */
    _onClickInvoiceExtractBox: function (ev) {
        ev.stopPropagation();
        var self = this;
        var box = ev.data.box;
        var field = this._invoiceExtractFields.getActiveField();
        if (!field) {
            return;
        }
        this._rpc({
            model: 'account.invoice',
            method: 'set_user_selected_box',
            args: [[this.state.res_id], box.getID()],
        }).then(function (fieldChangedInfo) {
            field.setSelectedBox(box);
            self._handleInvoiceExtractFieldChanged({
                fieldChangedInfo: fieldChangedInfo,
            });
        });
    },
    /**
     * Called when clicking on a box layer (excluding clicking on a box inside
     * the box layer). This unselects the selected box of the active field.
     *
     * @private
     * @param {OdooEvent} ev
     */
    _onClickInvoiceExtractBoxLayer: function (ev) {
        ev.stopPropagation();
        var self = this;
        var field = this._invoiceExtractFields.getActiveField();
        if (!field) {
            return;
        }
        var box = field.getSelectedBox();
        if (!box) {
            return;
        }
        field.unselectBox();
        this._rpc({
            model: 'account.invoice',
            method: 'remove_user_selected_box',
            args: [
                [this.state.res_id],
                box.getID()
            ],
        }).then(function (fieldChangedInfo) {
            self._handleInvoiceExtractFieldChanged({
                fieldChangedInfo: fieldChangedInfo,
            });
        });
    },
    /**
     * Called by the box that has been chosen by the OCR. Notify the
     * corresponding field that tracks selected boxes. This is useful in order
     * to determine which box should be selected. Indeed, an OCR chosen box
     * without any user-selected box should be marked as selected.
     *
     * @private
     * @param {OdooEvent} ev
     * @param {account_invoice_extract.Box} ev.data.box
     */
    _onChoiceOcrInvoiceExtractBox: function (ev) {
        ev.stopPropagation();
        var box = ev.data.box;
        var field = this._invoiceExtractFields.getField({
            name: box.getFieldName(),
        });
        if (!field) {
            return;
        }
        field.setOcrChosenBox(box);
    },
    /**
     * Called when a box is destroyed. This occurs in particular when there
     * is a change of mode from 'edit' to 'readonly', in which box layers and
     * boxes should not appear on the attachment anymore. In this case, the
     * fields should untrack the boxes.
     *
     * @private
     * @param {OdooEvent} ev
     * @param {account_invoice_extract.Box} ev.data.box
     */
    _onDestroyInvoiceExtractBox: function (ev) {
        ev.stopPropagation();
        var box = ev.data.box;
        if (!box.isOcrChosen() && !box.isSelected()) {
            // ignore untracked boxes
            return;
        }
        var field = this._invoiceExtractFields.getField({
            name: box.getFieldName(),
        });
        if (!field) {
            return;
        }
        field.unsetBox(box);
    },
    /**
     * Called by the box that has been selected by the user. Notify the
     * corresponding field that tracks selected boxes.
     *
     * @private
     * @param {OdooEvent} ev
     * @param {account_invoice_extract.Box} ev.data.box
     */
    _onSelectInvoiceExtractBox: function (ev) {
        ev.stopPropagation();
        var box = ev.data.box;
        var field = this._invoiceExtractFields.getField({
            name: box.getFieldName(),
        });
        if (!field) {
            return;
        }
        field.setSelectedBox(box);
    },
});

return InvoiceExtractFormRenderer;

});
    
odoo.define('account_invoice_extract.FormView', function (require) {
"use strict";

/**
 * This subview adds features related to OCR on attachments. An attachment
 * in the chatter can be sent to the OCR, which will add boxes per fields on
 * the attachment. Visually, the attachment preview displays boxes that are
 * generated by the OCR, and clicking on the boxes updates the form accordingly.
 * @see account_invoice_extract.FormRenderer
 */
var InvoiceExtractFormRenderer = require('account_invoice_extract.FormRenderer');

var FormView = require('web.FormView');
var view_registry = require('web.view_registry');

var InvoiceExtractFormView = FormView.extend({
    config: _.extend({}, FormView.prototype.config, {
        Renderer: InvoiceExtractFormRenderer,
    }),
});

view_registry.add('account_invoice_extract_preview', InvoiceExtractFormView);

return InvoiceExtractFormView;

});

odoo.define('account_invoice_extract.ThreadField', function (require) {
"use strict";

var ThreadField = require('mail.ThreadField');
require('mail_enterprise.ThreadField');

ThreadField.include({

    //--------------------------------------------------------------------------
    // Private
    //--------------------------------------------------------------------------

    /**
    * Override the thread rendering to warn the FormRenderer about attachments.
    * This is used by the FormRenderer to display an attachment preview.
    *
    * @override
    * @private
    */
    _fetchAndRenderThread: function () {
        var self = this;
        return this._super.apply(this, arguments).then(function () {
            if (self._threadWidget.attachments.length) {
                self.trigger_up('preview_attachment_validation');
            }
        });
    },
});

});
    
